######################################################################
### Projekt       : GartenLeistungen II                            ###
### Beschreibung  : Choice Model mit Apollo-Paket                  ###
###                 Conditional Logit im Preference Space          ###
### Output        : Flextable (Wordfile)                           ###
### Datum         : 11.07.2023                                     ###
### R-Version     : 4.3.1                                          ###
### Apollo Version: 0.2.9                                          ###
######################################################################


### Environment loeschen
rm(list=ls())

### Arbeitspfad festlegen
wd <- "S:/10 GIS/GartenLeistungen/Auswertung GartenLeistungen/R_GL2/pilot"
setwd(wd)

### Pakete laden (wenn nicht vorhanden, installieren)
if(!require(pacman)){ 
  install.packages("pacman")
  library(pacman)
}

p_load(apollo, flextable, magrittr, officer, haven, dplyr, stringr, readr, sjlabelled)

# ################################################################# #
#### INITIALISE APOLLO AND LOAD DATA                             ####
# ################################################################# #

### initialise apollo and core settings
apollo_initialise()
apollo_control= list (
  modelName = "Clogit",
  modelDescr ="Conditional Logit in preference space",
  indivID = "ID",
  mixing = FALSE # True waere Mixed Logit
)


# Dataframe für Flextable erstellen
dat <- data.frame(matrix(NA,nrow=30,ncol=2))

dat[,1] <- c("asc\nEntstehung eines Referenzparks in der Nachbarschaft", "asc1",
             "Größe","1",
             "Entfernung","2",
             "Gemeinschaftsaktivitäten","Gemeinschaft1",
             "Kulturveranstaltungen","Kultur1",
             "Umweltbildungsangebote","Bildung1",
             "Pflegeintensität\nje Stufe besserer Pflege","3",
             "Pflegeziele\nmit Vielfalt im Vordergund","4",
             "Toiletten","toilet1",
             "Spiel- und Sportgeräte","Spielsport1",
             "Beitrag\npro 1 €","5",
             "Anzahl","LL(start)","LL(0)","LL(final)","Rho-square","Adj. Rho-square","AIC","BIC")
colnames(dat) <- c("Attribute","Praeferenz")

### Datensatz aus Aufbereitungs-Skript laden
load("02_apollodataF.RData")
# falls labelled data genutzt wird: (hier für Frankfurt aber nicht der Fall)
# for (col in names(apollodata_F)) {
#   apollodata_F[[col]] <- as.numeric(val_labels(apollodata_F[[col]]))
# }
database <- as.data.frame(apollodata_F)
database <- database[order(database$ID),]
#database <- database[-which(is.na(database$choice)),]

# ################################################################# #
#### DEFINE apollo_beta()                                        ####
# ################################################################# #

### Startwerte festlegen
apollo_beta=c(asc = 0,
              b_groesse = 0,
              b_entfernung = 0,
              b_gemeinschaft = 0,
              b_kultur = 0,
              b_umweltbildung = 0,
              b_toiletten = 0,
              b_spiel = 0,
              b_pflegeint = 0,
              b_pflegeziele = 0,
              b_beitrag = 0)

# ################################################################# #
#### DEFINE MODEL AND LIKELIHOOD FUNCTION                        ####
# ################################################################# #

### keine Parameter fix halten
apollo_fixed = c()

### validieren
apollo_inputs = apollo_validateInputs()

apollo_probabilities=function(apollo_beta, apollo_inputs, functionality="estimate"){
  
  ### Function initialisation: do not change the following three commands
  ### Attach inputs and detach after function exit
  apollo_attach(apollo_beta, apollo_inputs)
  on.exit(apollo_detach(apollo_beta, apollo_inputs))
  
  ### Create list of probabilities P
  P = list()
  
  ### List of utilities (later integrated in mnl_settings below)
  V = list()
  V[['alt1']] =  (b_groesse * GROESSE.1 + b_entfernung * ENTFERNUNG.1 +
                         b_gemeinschaft * GEMEINSCHAFTSAKTIVITAETEN.1 + b_kultur * KULTURVERANSTALTUNGEN.1 +
                         b_umweltbildung * UMWELTBILDUNG.1 + b_toiletten * TOILETTEN.1 +
                         b_spiel * SPIEL_SPORT.1 +
                         b_pflegeint * PFLEGEINTENSITAET.1 + b_pflegeziele * PFLEGEZIELE.1 + b_beitrag * BEITRAG.1)
  
  V[['alt2']] = (b_groesse * GROESSE.2 + b_entfernung * ENTFERNUNG.2 +
                         b_gemeinschaft * GEMEINSCHAFTSAKTIVITAETEN.2 + b_kultur * KULTURVERANSTALTUNGEN.2 +
                         b_umweltbildung * UMWELTBILDUNG.2 + b_toiletten * TOILETTEN.2 +
                         b_spiel * SPIEL_SPORT.2 +
                         b_pflegeint * PFLEGEINTENSITAET.2 + b_pflegeziele * PFLEGEZIELE.2 + b_beitrag * BEITRAG.2)
  
  V[['alt3']] = (asc + b_groesse * GROESSE.3 + b_entfernung * ENTFERNUNG.3 +
                         b_gemeinschaft * GEMEINSCHAFTSAKTIVITAETEN.3 + b_kultur * KULTURVERANSTALTUNGEN.3 +
                         b_umweltbildung * UMWELTBILDUNG.3 + b_toiletten * TOILETTEN.3 +
                         b_spiel * SPIEL_SPORT.3 +
                         b_pflegeint * PFLEGEINTENSITAET.3 + b_pflegeziele * PFLEGEZIELE.3 + b_beitrag * BEITRAG.3)
  
  ### Define settings for MNL model component
  mnl_settings = list(
    alternatives  = c(alt1=1, alt2=2, alt3=3),
    avail         = 1, # all alternatives are available in every choice
    choiceVar     = choice,
    V             = V  # tell function to use list vector defined above
    #rows          = sample==i #not needed, für Frankfurt inkludieren wir alle
  )
  
  ### Compute probabilities using MNL model
  P[['model']] = apollo_mnl(mnl_settings, functionality)
  
  ### Take product across observation for same individual
  P = apollo_panelProd(P, apollo_inputs, functionality)
  
  ### Average across inter-individual draws - nur bei Mixed Logit!
  #P = apollo_avgInterDraws(P, apollo_inputs, functionality)
  
  ### Prepare and return outputs of function
  P = apollo_prepareProb(P, apollo_inputs, functionality)
  return(P)
}


# ################################################################# #
#### MODEL ESTIMATION                                            ####
# ################################################################# #

model = apollo_estimate(apollo_beta, apollo_fixed,
                        apollo_probabilities, apollo_inputs)

### model estmates speichern?
ps_model <- model
save(ps_model, file = "04_CL_PS_F_estimates.RData")

# ################################################################# #
#### MODEL OUTPUTS                                               ####
# ################################################################# #


### relevante variablen aus Model-Output extrahieren
z <- length(model$estimate)
est <- c(model$estimate[1:z-1]*10,model$estimate[z]/10)
se <- c(model$se[1:z-1]*10,model$se[z]/10)
t1 <- est-1.64*se
t2 <- est+1.64*se
n <- model$nIndivs
## stattdessen: output in datei speichern, und danach löschen:
apollo_saveOutput(model, saveOutput_settings = list(printPVal = 1))
s <- read_csv("Clogit_estimates.csv", show_col_types = FALSE)
file.remove(c("Clogit_estimates.csv", "Clogit_model.rds", "Clogit_output.txt", "Clogit_iterations.csv"))
p <- s$`Rob.p-val(0)`/2 # divide by 2 to get the one sided robust p value


nParams     <- length(model$apollo_beta)
nFreeParams <- nParams
if(!is.null(model$apollo_fixed)) nFreeParams <- nFreeParams - length(model$apollo_fixed)

# Tabelle erstellen (Umweg über df nicht unbedingt nötig, aber so ähnlich wie bei LBS)
df <- as.data.frame(matrix(data=NA,nrow=z*2+8,ncol=2))
rownames(df) <- dat[,1]

# Schleife fuer Signifikanz-Sterne
p_wert <- function(p){
  if(p < 0.01){print("***")}
  else{if(p < 0.05){print("**")}
    else{if(p < 0.1){print("*")}
      else{print("")}}}
}


# Tabelle auffuellen, alles auf zwei Nachkommastellen runden
for(j in 1:z){
  df[j*2-1,1] <- paste0(round(est[j], 2)," [",round(t1[j],2),";",round(t2[j],2),"]",p_wert(p[j]))
  df[j*2,1] <- paste0("(",round(se[j],2),")")
  if(j == z){
    x <-as.numeric(paste0(format(est[j], digits=2, nsmall = 2))) # Koeffizient des Beitrages
    y <- match(TRUE,round(x, 1:20) == x) # Nachkommastellen des Beitragskoeffizienten
    df[j*2-1,1] <- paste0(x," [",format(round(t1[j], digits=y), scientific = FALSE),";",format(round(t2[j], digits=y), scientific = FALSE),"]",p_wert(p[j]))
    df[j*2,1] <- paste0("(",format(round(se[j], digits=y), scientific = FALSE),")")
  }
}
rm(j)


# Model measures fuer Fusszeile in Tabelle ergaenzen 
df[z*2+1,1] <- n
df[z*2+2,1] <- round(model$LLStart,4)
df[z*2+3,1] <- round(model$LL0,4)
df[z*2+4,1] <- round(model$LLout,4)
df[z*2+5,1] <- round(1-(model$maximum/model$LL0),4)
df[z*2+6,1] <- round(1-((model$maximum-nFreeParams)/model$LL0),4)
df[z*2+7,1] <- round(-2*model$maximum + 2*nFreeParams,2)
df[z*2+8,1] <- round(-2*model$maximum + nFreeParams*log(model$nObs),2)

dat[,2] <- df[,1]

# parameter speichern?
# save(apollo_control,database,model,apollo_beta,apollo_fixed,apollo_probabilities,apollo_inputs,
#      est,se,n,file=paste("04_clogit_parameter_sample",i,".RData",sep=""))


### Dezimalpunkte in Kommas umwandeln
dat <- dat %>% 
  mutate(across(.cols= c(2:ncol(dat)),.fns = ~str_replace_all(., "\\.", "\\,"))) 



# ################################################################# #
#### Flextable                                                   ####
# ################################################################# #

### Einstellungen fuer Tabellenkopf und Fusszeile
my_header <- data.frame(col_keys=colnames(dat),
                        line1 = c("Model Results - Conditional logit"),
                        line2 = c("Attribute","Praeferenz"),
                        stringsAsFactors=FALSE)

my_footer <- data.frame(col_keys=colnames(dat),
                        line1 = c("*** 0,01 ; ** 0,05 ; * 0,1"),
                        line2 = c("Referenzpark: 0 m Entfernung, keine Gemeinschaftsaktivitäten, 
                        keine Kulturveranstaltungen, keine Umweltbildungsangebote, 
                        keine Toiletten, keine Spiel- und Sportgeräte, Basispflege, 
                                  Vielfalt und Nutzung sind gleich gewichtet"),
                        stringsAsFactors=FALSE)

### graphische Einstellungen fuer Flextable
flex <- flextable(dat) %>%
  theme_booktabs() %>%
  set_header_df(mapping = my_header, key="col_keys") %>%
  set_footer_df(mapping = my_footer, key="col_keys") %>%
  border(i=2,border.bottom=fp_border(color="black",width=1),part="header") %>%
  fontsize(size=12,i=1,part="header") %>%
  align(align="center",part="header") %>%
  align(align="right",part="footer") %>%
  border(border.top=fp_border(color="black",width=1),part="footer") %>%
  border(i=23,border.top=fp_border(width=1)) %>%
  bold(j=1,part="body") %>%
  bold(i=1:2,part="header") %>%
  font(fontname="Calibri",part="body") %>%
  font(fontname="Calibri",part="header") %>%
  merge_h(part="footer") %>%
  merge_at(i =1:2,j=1,part="body") %>%
  merge_at(i =3:4,j=1,part="body") %>%
  merge_at(i =5:6,j=1,part="body") %>%
  merge_at(i =7:8,j=1,part="body") %>%
  merge_at(i =9:10,j=1,part="body") %>%
  merge_at(i =11:12,j=1,part="body") %>%
  merge_at(i =13:14,j=1,part="body") %>%
  merge_at(i =15:16,j=1,part="body") %>%
  merge_at(i =17:18,j=1,part="body") %>%
  merge_at(i =19:20,j=1,part="body") %>%
  merge_at(i =21:22,j=1,part="body") %>%
  merge_at(i=1,part="header") %>%
  width(j=2,width=2.5) %>%
  width(j=1,width=4)

#----------------------------------------------
# Tabelle exportieren
#----------------------------------------------

### Vorschau in R:
print(flex)

### Oeffnen und Speichern in Word:
#print(flex,preview="docx")
#set_prop <- prop_section(page_size = page_size(orient = "landscape"))
save_as_docx(flex, path = "04_clogit_F_Konfidenzintervall_preference_space.docx")#, pr_section = set_prop)
