#include "utils.h"
#include "mainwindow.h"

#include "../obj/ui_mainwindow.h"
#include <QDebug>
#include <QKeyEvent>

#include <unistd.h>
#include <iostream>

using namespace VmbCPP;


MainWindow::MainWindow(QWidget *parent)
: QMainWindow(parent)
, ui(new Ui::MainWindow)
, sys(VmbSystem::GetInstance()) // Create and get Vimba singleton
, curCam(CameraPtr())
, cameras(CameraPtrVector())
, camInfoNames({"name","model","id","serial","interface","state"})
{
	ui->setupUi(this);
	ui->tw_cams->setColumnCount(camInfoNames.size());
	ui->tw_cams->setHorizontalHeaderLabels(camInfoNames);

	QObject::connect(ui->b_update, SIGNAL(clicked()), this, SLOT(updateCams()));
	QObject::connect(ui->b_openclose, SIGNAL(clicked()), this, SLOT(openCloseCam()));
	QObject::connect(ui->tw_cams, SIGNAL(cellClicked(int,int)), this, SLOT(selectCam(int,int)));

	log(getVersionString());

	if ( VmbErrorSuccess == sys.Startup() )
	{
	  log( " Started Vimba C++ Api" );
	  updateCams();
	}
	else
	{
		log( " Failed to start Vimba C++ Api" );
	}
}

MainWindow::~MainWindow()
{
	if ( VmbErrorSuccess == sys.Shutdown() )
	{
	  log( " Shutdown Vimba Api" );
	}
	else
	{
		log( " Failed to Shutdown Vimba Api" );
	}
  delete ui;
}

void MainWindow::keyPressEvent(QKeyEvent *e)
{
	//will not work on other windows (eg floating dockwidgets)
	if ( e->key() == Qt::Key_Escape )
	{
		qApp->quit();
	}
	return QWidget::keyPressEvent(e);
}

void MainWindow::selectCam(int row,int )
{
	curCam = cameras.at(row);
	//display Stream on label (stackedWidget)
	//display options in dockwidget-right

}

void MainWindow::openCloseCam()
{

	QPushButton* b = ui->b_openclose;
	if( b->text() == "open" )
	{
		auto ret = curCam->Open(VmbAccessModeFull);
		// log( QString("foo"));
		if( ret == VmbErrorSuccess )
			b->setText("close");
		log( QString("Open camera"), ret );
	}
	else if( b->text() == "close" )
	{
		auto ret = curCam->Close();
		if( ret == VmbErrorSuccess )
			b->setText("open");
		log( QString("Close camera"), ret );
	}
}

void MainWindow::updateCams()
{
	// clear
	ui->tw_cams->setRowCount(0);
	ui->tw_cams->clearContents(); //dont remove headers
	curCam = CameraPtr();
	cameras = CameraPtrVector();

	// get new
	VmbErrorType err = sys.GetCameras( cameras ); // fetch all cameras known to Vimba
	if( VmbErrorSuccess == err )
	{
		// query and print all _static_ details of all known cams [without opening cams]
		// auto ncam = cameras.size();
		// log( QString("Found %1 Camera%2!").arg(ncam).arg(ncam>1?"s":""));


		if( cameras.size() )
			ui->tw_cams->setCurrentCell(0,0); //select row
	}
	else
	{
		log( QString("Could not list cameras."), err);
	}
}

/// get static cam info
void MainWindow::getCamInfo( const CameraPtr &camera )
{
	std::string str;
	QStringList infos;

	infos << "closed"; //initially cams are closed, right?

	addRow(infos);

}

///add row to listwidget
void MainWindow::addRow( QStringList& camInfos )
{
	auto row = ui->tw_cams->rowCount();
	ui->tw_cams->setRowCount(row+1);
	for( auto col=0; col < ui->tw_cams->columnCount(); col++)
	{
		auto item = new QTableWidgetItem( camInfos.at(col) );
		item->setTextAlignment(Qt::AlignCenter);
		item->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable); //but not editable!

		ui->tw_cams->setItem( row, col, item );
	}
}

void MainWindow::log(QString msg, VmbError_t err )
{
	msg.append(QString("%1").arg(errorCodeToMessage(err)));

	//for std::string, use: log(QString::fromStdString(msg))
	auto idx = ui->lw_log->count()-1;
	ui->lw_log->insertItem( idx , msg );
	qDebug() << msg;
}



	//XXX use sth like isClosed() instead??

	//XXX use icon instead of text
	//QTableWidgetItem(const QIcon &icon, const QString &text, int type = Type)

	// addRow(infos);

	// random rows for mockup testing when no cam is avaiable
	// QStringList sl= {randomString(),randomString(),randomString(),randomString(),randomString(),randomString()};
	// addRow(sl);
